/**
* \file: gst_viv_renderer.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* GL Render Plugin using Vivante-Extensions for mapping
* YUV video streams to textures
*
* \component: gst_viv_demo
*
* \author: Michael Methner ADITG/SW1 mmethner@de.adit-jv.com
*
* \copyright: (c) 2003 - 2012 ADIT Corporation
*
* \history
* 0.1 Michael Methner Initial version
*
***********************************************************************/
#include "gst_viv_renderer.h"

#include <gst/video/video.h>
#include <glib.h>
#include <string.h>
#include <stdio.h>

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>
#include <HAL/gc_hal_eglplatform.h>

#include <linux/videodev2.h>

#include <gst/base/gstpushsrc.h>
#include <gst/fsl/gstbufmeta.h>
#include "gst_egl_buffer.h"


/* PRQA: Lint Message 160, 826: deactivation because casting mechanism of GObject throws the finding */
/*lint -e826 -e160*/



#ifndef PACKAGE
#define PACKAGE "gst_viv_renderer"
#endif


#define DEFAULT_TEXTURE_FORMAT GST_VIV_RENDERER_TEX_FORMAT_AUTODETECT
#define DEFAULT_TEXTURE_TILED FALSE
#define DEFAULT_LOOPING_COUNT 1
#define DEFAULT_TIMESTAMP_INCR GST_CLOCK_TIME_NONE


enum
{
  PROP_TEXTURE_FORMAT = 1,
  PROP_TEXTURE_TILED,
  PROP_LOOPING_COUNT,
  PROP_TIMESTAMP_INCR
};


enum
{
  SIGNAL_RENDER,
  LAST_SIGNAL
};


static guint gst_viv_renderer_signals[LAST_SIGNAL] = { 0 };


static GstStaticPadTemplate sink_factory =
  GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (
        GST_VIDEO_CAPS_YUV ("{ I420, YV12 , YUY2, YUYV, UYVY, NV12, NV21 }")
        )
  );

static GstStaticPadTemplate src_factory =
  GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY
  );


GST_DEBUG_CATEGORY_STATIC (gst_viv_renderer_debug);
#define GST_CAT_DEFAULT gst_viv_renderer_debug


/* PRQA: Lint Message 19, 123, 144, 751 : deactivation because macro is related to GStreamer framework */
/*lint -e19 -e123 -e144 -e751 */
GST_BOILERPLATE (GstVivRenderer,gst_viv_renderer, GstElement,
                GST_TYPE_ELEMENT);
/*lint +e19 +e123 +e144 +e751 */


static void gst_viv_renderer_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_viv_renderer_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

static GstStateChangeReturn
gst_viv_renderer_change_state (GstElement *element, GstStateChange transition);

static GstFlowReturn gst_viv_renderer_chain(GstPad * pad, GstBuffer * buf);

static gboolean gst_viv_renderer_event(GstPad * pad, GstEvent *event);

static void
gst_viv_renderer_finalize(GObject * object);

gboolean
gst_viv_renderer_setcaps(GstPad *pad, GstCaps *caps);

static void
gst_viv_renderer_base_init (gpointer gclass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (gclass);

  gst_element_class_set_details_simple(element_class,
    "gst_viv_renderer",
    "Sink/Video/Device",
    "gst_viv_renderer",
    "Michael Methner mmethner@de.adit-jv.com");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_factory));

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_factory));
}


static void
gst_viv_renderer_class_init (GstVivRendererClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;


  gobject_class = (GObjectClass *) klass;
  gobject_class->finalize = gst_viv_renderer_finalize;
  gstelement_class = (GstElementClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_viv_renderer_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_viv_renderer_get_property);

  gstelement_class->change_state = GST_DEBUG_FUNCPTR (gst_viv_renderer_change_state);

  gst_viv_renderer_signals[SIGNAL_RENDER] =
       g_signal_new ("render",
           G_TYPE_FROM_CLASS (klass),
           G_SIGNAL_RUN_LAST,
           G_STRUCT_OFFSET (GstVivRendererClass, render),
           NULL, NULL,
           gst_marshal_VOID__OBJECT,
           G_TYPE_NONE, 1,
           GST_TYPE_PAD);

  g_object_class_install_property (gobject_class, PROP_TEXTURE_FORMAT,
       g_param_spec_uint("texture_format", "texture_format",
           "Texture color format to use for mapping incoming frame to the texture",
           0x00000000, 0xFFFFFFFF, DEFAULT_TEXTURE_FORMAT, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_TEXTURE_TILED,
       g_param_spec_boolean("texture_tiled", "texture_tiled",
           "Map incoming texture as an tiled image",
           DEFAULT_TEXTURE_TILED, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_LOOPING_COUNT,
       g_param_spec_uint("looping_count", "looping_count",
           "Number of times of calls of the render_callback for each incoming frame",
           1,G_MAXUINT, DEFAULT_LOOPING_COUNT, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_TIMESTAMP_INCR,
       g_param_spec_uint64("timestamp_incr", "timestamp_incr",
           "Increment of each timestamp when the looping feature is activated",
           0, GST_CLOCK_TIME_NONE, DEFAULT_TIMESTAMP_INCR, G_PARAM_READWRITE));
}

static void
gst_viv_renderer_tex_free(GLuint texname)
{
  GST_LOG("delete texture %ld", texname);
  glDeleteTextures(1, &texname);
}

static void
gst_viv_renderer_finalize(GObject * object)
{
  GstVivRenderer * renderer = GST_VIV_RENDERER(object);
  g_mutex_free(renderer->pool_lock);
  g_cond_free(renderer->pool_data);

  g_mutex_free(renderer->buffer_mutex);
  renderer->pool_lock = NULL;
  renderer->pool_data = NULL;
  g_hash_table_unref(renderer->tex_id);
}


static void
gst_viv_renderer_init (GstVivRenderer * renderer,
    GstVivRendererClass * gclass)
{
  GstPadTemplate *sink_pad_template;
  GstPadTemplate *src_pad_template;

  gclass = gclass;

  renderer->pool_lock = g_mutex_new();
  renderer->pool_data = g_cond_new();
  renderer->buffer_pool = NULL;

  renderer->buffer_mutex = g_mutex_new();
  renderer->buffer = NULL;

  sink_pad_template = gst_element_class_get_pad_template(GST_ELEMENT_CLASS(
      gclass), "sink");

  renderer->sinkpad = gst_pad_new_from_template(sink_pad_template, "sink");
  gst_pad_use_fixed_caps(renderer->sinkpad);
  gst_pad_set_chain_function(renderer->sinkpad, GST_DEBUG_FUNCPTR(
      gst_viv_renderer_chain));
  gst_pad_set_event_function(renderer->sinkpad, GST_DEBUG_FUNCPTR(
      gst_viv_renderer_event));
  gst_pad_set_setcaps_function(renderer->sinkpad, GST_DEBUG_FUNCPTR(
      gst_viv_renderer_setcaps));

  /* create source pad */
  /* src template has to be defined in subclass */
  src_pad_template = gst_element_class_get_pad_template(GST_ELEMENT_CLASS(
      gclass), "src");
  if (src_pad_template == NULL)
    {
      GST_ERROR("Could not get source pad template");
      return;
    }
  renderer->srcpad = gst_pad_new_from_template(src_pad_template, "src");

  gst_pad_set_event_function(renderer->srcpad, GST_DEBUG_FUNCPTR(
      gst_viv_renderer_event));
  gst_pad_use_fixed_caps(renderer->srcpad);

  gst_element_add_pad(GST_ELEMENT(renderer), renderer->sinkpad);
  gst_element_add_pad(GST_ELEMENT(renderer), renderer->srcpad);

  renderer->src_width = 0;
  renderer->src_height = 0;
  renderer->texture_format = DEFAULT_TEXTURE_FORMAT;
  renderer->texture_tiled = DEFAULT_TEXTURE_TILED;
  renderer->looping_count = DEFAULT_LOOPING_COUNT;
  renderer->timestamp_incr = GST_CLOCK_TIME_NONE;
  renderer->timestamp = GST_CLOCK_TIME_NONE;

  renderer->tex_id = g_hash_table_new_full (g_direct_hash, g_direct_equal,
                                          NULL,
                                          (GDestroyNotify) gst_viv_renderer_tex_free);
  renderer->running = FALSE;
}



static void
gst_viv_renderer_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstVivRenderer * renderer = GST_VIV_RENDERER(object);
  value = value;
  switch (prop_id) {
    case PROP_TEXTURE_FORMAT:
        renderer->texture_format = g_value_get_uint(value);
        break;
    case PROP_TEXTURE_TILED:
        renderer->texture_tiled = g_value_get_boolean(value);
        break;
    case PROP_LOOPING_COUNT:
      renderer->looping_count = g_value_get_uint(value);
      break;
    case PROP_TIMESTAMP_INCR:
      renderer->timestamp_incr = g_value_get_uint64(value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static void
gst_viv_renderer_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstVivRenderer * renderer = GST_VIV_RENDERER(object);
  value = value;
  switch (prop_id) {
    case PROP_TEXTURE_FORMAT:
      g_value_set_uint(value, renderer->texture_format);
      break;
    case PROP_TEXTURE_TILED:
      g_value_set_boolean(value, renderer->texture_tiled);
      break;
    case PROP_LOOPING_COUNT:
      g_value_set_uint(value, renderer->looping_count);
      break;
    case PROP_TIMESTAMP_INCR:
      g_value_set_uint64(value, renderer->timestamp_incr);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static GstStateChangeReturn
gst_viv_renderer_change_state(GstElement *element, GstStateChange transition)
{
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
  GstVivRenderer * renderer = GST_VIV_RENDERER(element);

  GST_LOG("switch state: transition: %d current_state: %d", transition,element->current_state);

  switch (transition)
  {
    case GST_STATE_CHANGE_NULL_TO_READY:
       break;
    case GST_STATE_CHANGE_READY_TO_PAUSED:
          renderer->running = TRUE;
          break;
    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
          break;
    default:
        break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state(element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
  {
    GST_ERROR("state change failure");
    return ret;
  }

  switch (transition)
  {
    case GST_STATE_CHANGE_READY_TO_NULL:
      {
        g_hash_table_remove_all (renderer->tex_id);
        g_mutex_lock(renderer->pool_lock);
        while (renderer->buffer_pool)
          {
            GstBuffer * buffer = (GstBuffer *) renderer->buffer_pool->data;
            renderer->buffer_pool = g_slist_delete_link(renderer->buffer_pool,
                renderer->buffer_pool);
            gst_buffer_unref(buffer);
          }
        g_mutex_unlock(renderer->pool_lock);

        if(renderer->src_caps)
          {
            gst_caps_unref(renderer->src_caps);
            renderer->src_caps = NULL;
          }
      }
      break;
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      renderer->running = FALSE;
      if(renderer->buffer)
        {
          /* unref this buffer already in PAUSED TO READY state
           * because otherwise mfw_v4lsrc will delete it before
           * this plugin can release the buffer
           */
          gst_buffer_unref(renderer->buffer);
          renderer->buffer = NULL;
        }
      break;
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      break;
    default:
      break;
  }
  return ret;
}


static gboolean
gst_viv_renderer_event(GstPad * pad, GstEvent *event)
{
  pad = pad;
  GST_LOG("received: %s", GST_EVENT_TYPE_NAME(event));

  return TRUE;
}


GstClockTime gst_viv_renderer_get_current_timestamp(GstVivRenderer * renderer)
{
  return renderer->timestamp;
}

GLuint gst_viv_renderer_get_current_texname(GstVivRenderer * renderer)
{
  GLuint texname = 0;
  g_mutex_lock(renderer->buffer_mutex);
  if(renderer->buffer)
    {
      GstBufferMeta * meta = GST_BUFFER_META(renderer->buffer->_gst_reserved[G_N_ELEMENTS(renderer->buffer->_gst_reserved)-1]);
      if (meta != NULL)
        {
          guint8 * logical = renderer->buffer->data;
          guint8 * physical = (guint8 *)meta->physical_data;
          texname = (GLuint)g_hash_table_lookup (renderer->tex_id, physical);
          if (texname != 0)
            {
                g_mutex_unlock(renderer->buffer_mutex);
                GST_LOG("return hashed texture %ld", texname);
                return texname;
            }

          if (G_UNLIKELY (g_hash_table_size (renderer->tex_id) == MAX_VIV_BUFFERS))
            {
                g_mutex_unlock(renderer->buffer_mutex);
                GST_ERROR("texture id from v4l out of range");
                return 0;
            }

            {
              glGenTextures(1, &texname);
              glBindTexture(GL_TEXTURE_2D, texname);
              GST_LOG("create new texture %ld", texname);
              if (renderer->texture_tiled)
                {
                  glTexDirectTiledMapVIV(GL_TEXTURE_2D,
                      renderer->src_width, renderer->src_height,
                      renderer->texture_format, (void**) &logical,
                      (GLuint *) &physical);
                }
              else
                {
                  glTexDirectVIVMap(GL_TEXTURE_2D, renderer->src_width,
                      renderer->src_height, renderer->texture_format,
                      (void**) &logical, (GLuint *) &physical);
                }
            }
            g_hash_table_insert (renderer->tex_id, physical, (void*)texname);
        }
      else
        {
           GST_ERROR("No meta buffer found");
           texname = 0;
        }
    }
  else
    {
      GST_ERROR("no buffer available");
      texname = 0;
    }
  g_mutex_unlock(renderer->buffer_mutex);

  return texname;
}


gboolean
gst_viv_renderer_setcaps(GstPad *pad, GstCaps *caps)
{
  gboolean retval= TRUE;
  GstStructure *s;
  GstElement * element = GST_PAD_PARENT(pad);
  GstVivRenderer * renderer = GST_VIV_RENDERER(element);

  s = gst_caps_get_structure(caps, 0);

  retval = retval & gst_structure_get_int(s, "width", &renderer->src_width);
  retval = retval & gst_structure_get_int(s, "height", &renderer->src_height);
  if(!retval)
    {
      GST_ERROR("could not parse width and height");
    }

  if(renderer->texture_format == GST_VIV_RENDERER_TEX_FORMAT_AUTODETECT)
    {
      guint32 fourcc=0;
      gst_structure_get_fourcc (s, "format", &fourcc);
      switch (fourcc)
        {
        case GST_MAKE_FOURCC('I','4','2','0'):
          /* i420 is not supported natively by Vivante API and requires
           * color compensation in the shader (swap U and V).
           * fall through to YV12 */
        case GST_MAKE_FOURCC('Y','V','1','2'):
          renderer->texture_format=GL_VIV_YV12;
          break;
        case GST_MAKE_FOURCC('N','V','1','2'):
          renderer->texture_format = GL_VIV_NV12;
          break;
        case GST_MAKE_FOURCC('Y','U','Y','2'):
        case GST_MAKE_FOURCC('Y','U','Y','V'):
          renderer->texture_format = GL_VIV_YUY2;
          break;
        case GST_MAKE_FOURCC('U','Y','V','Y'):
          renderer->texture_format = GL_VIV_UYVY;
          break;
        case GST_MAKE_FOURCC('N','V','2','1'):
          renderer->texture_format = GL_VIV_NV21;
          break;
        default:
          GST_ERROR("unsupported color format");
          retval = FALSE;
        }
    }

  return retval;
}


/**
 * \func gst_viv_renderer_finalize_buffer
 *
 * This function is called whenever the reference count of a GstVivBuffer
 * drops to zero. The function increments the refcount again and pushes it
 * into the sink queue for reuse
 *
 * \param buffer Buffer to be recycled
 * \param element Pointer the decoder which has created the buffer
 *
 * \return True on success, false on failure
 *
 */
gboolean
gst_viv_renderer_finalize_buffer(GstEglBuffer * buffer, GstElement * element)
{
  gboolean recycled = FALSE;
  GstVivRenderer * sink = GST_VIV_RENDERER(element);

  if(sink->running)
    {
      GST_LOG("lock: buf: %p data: %p",buffer, GST_BUFFER(buffer)->data);
      g_mutex_lock(sink->pool_lock);
      sink->buffer_pool = g_slist_append(sink->buffer_pool, buffer);
      g_cond_signal(sink->pool_data);
      g_mutex_unlock(sink->pool_lock);
      GST_LOG("unlock");
      gst_buffer_ref(GST_BUFFER(buffer));
      recycled = TRUE;
    }

  return recycled;
}

/**
 * \func gst_viv_renderer_buffer_new
 *
 * Creates new eglbuffer.
 * Set the finalize fncptr of the buffer to returned to the sink
 * queue when unreferenced
 *
 * \param decoder "this" pointer
 *
 * \return True on success, false on failure
 *
 */
static GstEglBuffer *
gst_viv_renderer_buffer_new(GstVivRenderer *renderer)
{
  GstEglBuffer *buffer;
  buffer = GST_EGL_BUFFER(gst_mini_object_new (GST_TYPE_EGL_BUFFER));

  buffer->parent = gst_object_ref(renderer);
  buffer->finalize = &gst_viv_renderer_finalize_buffer;

  GST_LOG("created new GstVivBuffer: %p", buffer);
  return buffer;
}




static GstFlowReturn
gst_viv_renderer_chain(GstPad * sink, GstBuffer * buf)
{
  guint i;
  GstFlowReturn retval = GST_FLOW_OK;
  GstElement * element = GST_PAD_PARENT(sink);
  GstVivRenderer * renderer = GST_VIV_RENDERER(element);
  guint loop_count = renderer->looping_count;

  GST_LOG("start: buf: %p data: %p size:%d timestamp: %" GST_TIME_FORMAT " duration: %" GST_TIME_FORMAT "",
          buf, buf->data, buf->size, GST_TIME_ARGS(buf->timestamp), GST_TIME_ARGS(buf->duration));

  g_mutex_lock(renderer->buffer_mutex);
  if(renderer->buffer)
    {
      gst_buffer_unref(renderer->buffer);
    }
  renderer->buffer = buf;
  g_mutex_unlock(renderer->buffer_mutex);

  for(i=0;i<loop_count && retval == GST_FLOW_OK;i++)
    {
      GstClockTime duration;

      g_mutex_lock(renderer->pool_lock);
      while (!renderer->buffer_pool)
        {
          GST_LOG("waiting for buffer ...");
          g_cond_wait(renderer->pool_data, renderer->pool_lock);
        }
      GST_LOG("got buffer");

      GstEglBuffer * vivBuffer = (GstEglBuffer *) renderer->buffer_pool->data;
      renderer->buffer_pool = g_slist_delete_link(renderer->buffer_pool,
          renderer->buffer_pool);
      g_mutex_unlock(renderer->pool_lock);

      eglMakeCurrent(renderer->eglDisplay, vivBuffer->window_surface,
          vivBuffer->window_surface, renderer->eglContext);

      if(renderer->timestamp_incr == GST_CLOCK_TIME_NONE)
            {
              duration = GST_BUFFER_DURATION(buf) / loop_count;
            }
          else
            {
              duration = renderer->timestamp_incr;
            }

      GST_BUFFER_TIMESTAMP(vivBuffer) = GST_BUFFER_TIMESTAMP(buf) + i * duration;
      GST_BUFFER_DURATION(vivBuffer) = duration;
      renderer->timestamp = GST_BUFFER_TIMESTAMP(vivBuffer);

      g_signal_emit(element, gst_viv_renderer_signals[SIGNAL_RENDER], 0,
          renderer->sinkpad);

      GST_LOG("pushing: %p ( %p)", vivBuffer, GST_BUFFER(vivBuffer)->data);
      retval = gst_pad_push(renderer->srcpad, GST_BUFFER(vivBuffer));
    }

  g_mutex_lock(renderer->buffer_mutex);
  if(renderer->buffer)
    {
      gst_buffer_unref(renderer->buffer);
      renderer->buffer = NULL;
    }
  g_mutex_unlock(renderer->buffer_mutex);

  return retval;
}


void
gst_viv_renderer_set_egl(GstVivRenderer * renderer,
    EGLDisplay eglDisplay,EGLContext eglContext)
{
  renderer->eglContext = eglContext;
  renderer->eglDisplay = eglDisplay;
}

gboolean
gst_viv_renderer_add_pixmap(GstVivRenderer * renderer,
   EGLSurface egl_window_surface, EGLNativeDisplayType native_display, EGLNativePixmapType native_pixmap)
{
  gctINT width;
  gctINT height;
  gctINT bpp;
  gctINT stride;
  gctPOINTER data;

  GstEglBuffer * buffer = gst_viv_renderer_buffer_new(renderer);

  gceSTATUS rv = gcoOS_GetPixmapInfo( native_display, native_pixmap,
        &width, &height, &bpp, &stride, &data);
  if(rv != gcvSTATUS_OK)
    {
      GST_ERROR("could not get pixmap information\n");
      return FALSE;
    }

  GST_BUFFER(buffer)->data = data;
  GST_BUFFER(buffer)->size = stride * height;

  buffer->native_pixmap = native_pixmap;
  buffer->window_surface = egl_window_surface;
  if(renderer->src_caps == NULL)
    {
      switch(bpp)
      {
      case 16:
          renderer->src_caps = gst_caps_new_simple("video/x-raw-rgb",
              "bpp", G_TYPE_INT, 16,
              "depth", G_TYPE_INT, 16,
              "width", G_TYPE_INT, width,
              "height", G_TYPE_INT, height,
              "framerate", GST_TYPE_FRACTION, 30,1,
              "endianness", G_TYPE_INT, 1234,
              "red_mask", G_TYPE_INT,   63488,
              "green_mask", G_TYPE_INT, 2016,
              "blue_mask", G_TYPE_INT,  31,
              "pixel-aspect-ratio",GST_TYPE_FRACTION,1,1,
              NULL);
          break;
      case 32:
        renderer->src_caps = gst_caps_new_simple("video/x-raw-rgb",
            "bpp", G_TYPE_INT, 32,
            "depth", G_TYPE_INT, 32,
            "width", G_TYPE_INT, width,
            "height", G_TYPE_INT, height,
            "framerate", GST_TYPE_FRACTION, 30, 1,
            "endianness", G_TYPE_INT, 4321,
            "red_mask", G_TYPE_INT,   0x0000FF00,
            "green_mask", G_TYPE_INT, 0x00FF0000,
            "blue_mask", G_TYPE_INT,  0xFF000000,
            "pixel-aspect-ratio", GST_TYPE_FRACTION, 1, 1,
            NULL);
        break;
      default:
        GST_ERROR("unsupported pixelformat");
        return FALSE;
      }
    }

  GST_BUFFER(buffer)->caps = gst_caps_ref(renderer->src_caps);

  g_mutex_lock(renderer->pool_lock);
  renderer->buffer_pool = g_slist_append(renderer->buffer_pool, buffer);
  g_cond_signal(renderer->pool_data);
  g_mutex_unlock(renderer->pool_lock);

  return TRUE;
}


static gboolean
plugin_init (GstPlugin * plugin)
{
  GST_DEBUG_CATEGORY_INIT (gst_viv_renderer_debug, "gst_viv_renderer",
      0, "Vivante OpenGL Renderer");

  return gst_element_register (plugin, "gst_viv_renderer", GST_RANK_PRIMARY,
      GST_TYPE_VIV_RENDERER);
 }


GST_PLUGIN_DEFINE (
    GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "gst_viv_plugin",
    "GST Vivante App sink",
    plugin_init,
    "0.0.2",
    "Proprietary",
    "ADIT",
    "http://TBD/"
)


/*lint +e826 +e160*/
